<?php
/**
 * Elevadex Review Management System
 * Admin Panel Entry Point
 * 
 * @version 2.0
 * @author Elevadex Development Team
 */

// Define application constant
define('ELEVADEX_APP', true);

// Initialize application
$app = require_once '../core/App.php';

// Check for maintenance mode
if ($app->isMaintenanceMode()) {
    $allowedIPs = $app->config('maintenance.allowed_ips', []);
    $clientIP = $_SERVER['REMOTE_ADDR'] ?? '';
    
    if (!in_array($clientIP, $allowedIPs)) {
        http_response_code(503);
        include '../public/templates/maintenance.php';
        exit;
    }
}

// Initialize authentication
require_once '../core/auth/Auth.php';
$auth = new Auth($app);

// Cleanup expired auth data periodically (1% chance)
if (rand(1, 100) === 1) {
    $auth->cleanup();
}

// Handle authentication requests
$loginError = '';
$loginSuccess = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        
        switch ($_POST['action']) {
            case 'login':
                // Verify CSRF token for security
                if ($app->config('security.csrf_protection') && 
                    !$auth->verifyCSRFToken($_POST['csrf_token'] ?? '')) {
                    $loginError = 'Security token mismatch. Please try again.';
                    break;
                }
                
                $password = $_POST['password'] ?? '';
                $remember = isset($_POST['remember']);
                
                $result = $auth->login($password, $remember);
                
                if ($result['success']) {
                    // Successful login - redirect to prevent form resubmission
                    header('Location: ' . $_SERVER['PHP_SELF'] . '?login=success');
                    exit;
                } else {
                    $loginError = $result['message'];
                }
                break;
                
            case 'logout':
                $auth->logout();
                header('Location: ' . $_SERVER['PHP_SELF'] . '?logout=success');
                exit;
                break;
        }
    }
}

// Handle GET parameters
if (isset($_GET['login']) && $_GET['login'] === 'success') {
    $loginSuccess = 'Welcome back! Login successful.';
}

if (isset($_GET['logout']) && $_GET['logout'] === 'success') {
    $loginSuccess = 'You have been logged out successfully.';
}

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    // Show login page
    include 'login.php';
    exit;
}

// User is authenticated - load admin panel
require_once '../core/database/JsonDatabase.php';
require_once '../core/database/BusinessModel.php';
require_once '../core/database/ReviewModel.php';
require_once '../core/database/AnalyticsModel.php';

// Initialize data models
$db = new JsonDatabase($app);
$businessModel = new BusinessModel($db);
$reviewModel = new ReviewModel($db);
$analyticsModel = new AnalyticsModel($db);

// Get current tab/page
$currentTab = $_GET['tab'] ?? 'dashboard';
$allowedTabs = ['dashboard', 'businesses', 'reviews', 'analytics', 'settings', 'qr-generator'];

if (!in_array($currentTab, $allowedTabs)) {
    $currentTab = 'dashboard';
}

// Handle AJAX requests
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    
    try {
        switch ($_GET['ajax']) {
            case 'dashboard_stats':
                $stats = [
                    'total_businesses' => $businessModel->count(),
                    'total_reviews' => $reviewModel->count(),
                    'pending_reviews' => $reviewModel->countPending(),
                    'this_month_reviews' => $reviewModel->countThisMonth(),
                ];
                echo json_encode(['success' => true, 'data' => $stats]);
                break;
                
            case 'recent_activity':
                $recentReviews = $reviewModel->getRecent(10);
                echo json_encode(['success' => true, 'data' => $recentReviews]);
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Unknown AJAX request']);
        }
    } catch (Exception $e) {
        $app->log('error', 'AJAX error: ' . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Server error']);
    }
    
    exit;
}

// Handle form submissions
$successMessage = '';
$errorMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['action'])) {
    // Verify CSRF token
    if ($app->config('security.csrf_protection') && 
        !$auth->verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errorMessage = 'Security token mismatch. Please refresh the page and try again.';
    } else {
        // Load appropriate controller based on current tab
        try {
            switch ($currentTab) {
                case 'businesses':
                    require_once 'controllers/BusinessController.php';
                    $controller = new BusinessController($app, $businessModel);
                    $result = $controller->handle($_POST);
                    break;
                    
                case 'reviews':
                    require_once 'controllers/ReviewController.php';
                    $controller = new ReviewController($app, $reviewModel);
                    $result = $controller->handle($_POST);
                    break;
                    
                case 'analytics':
                    require_once 'controllers/AnalyticsController.php';
                    $controller = new AnalyticsController($app, $analyticsModel);
                    $result = $controller->handle($_POST);
                    break;
                    
                case 'settings':
                    require_once 'controllers/SettingsController.php';
                    $controller = new SettingsController($app);
                    $result = $controller->handle($_POST);
                    break;
                    
                default:
                    $result = ['success' => false, 'message' => 'Invalid action'];
            }
            
            if ($result['success']) {
                $successMessage = $result['message'];
                // Redirect to prevent form resubmission
                header('Location: ' . $_SERVER['PHP_SELF'] . '?tab=' . $currentTab . '&success=' . urlencode($result['message']));
                exit;
            } else {
                $errorMessage = $result['message'];
            }
            
        } catch (Exception $e) {
            $app->log('error', 'Controller error: ' . $e->getMessage());
            $errorMessage = 'An error occurred while processing your request.';
        }
    }
}

// Handle success message from redirect
if (isset($_GET['success'])) {
    $successMessage = $_GET['success'];
}

// Get data for current tab
$tabData = [];
try {
    switch ($currentTab) {
        case 'dashboard':
            $tabData = [
                'businesses' => $businessModel->getAll(),
                'recent_reviews' => $reviewModel->getRecent(5),
                'stats' => [
                    'total_businesses' => $businessModel->count(),
                    'total_reviews' => $reviewModel->count(),
                    'pending_reviews' => $reviewModel->countPending(),
                    'this_month_reviews' => $reviewModel->countThisMonth(),
                ],
            ];
            break;
            
        case 'businesses':
            $tabData = [
                'businesses' => $businessModel->getAll(),
                'editing' => isset($_GET['edit']) ? $businessModel->getById($_GET['edit']) : null,
            ];
            break;
            
        case 'reviews':
            $filter = $_GET['filter'] ?? 'all';
            $search = $_GET['search'] ?? '';
            $tabData = [
                'reviews' => $reviewModel->getFiltered($filter, $search),
                'filter' => $filter,
                'search' => $search,
                'businesses' => $businessModel->getAll(),
            ];
            break;
            
        case 'analytics':
            $tabData = [
                'analytics' => $analyticsModel->getDashboardData(),
                'businesses' => $businessModel->getAll(),
            ];
            break;
            
        case 'qr-generator':
            $businessId = $_GET['business_id'] ?? null;
            $tabData = [
                'businesses' => $businessModel->getAll(),
                'selected_business' => $businessId ? $businessModel->getById($businessId) : null,
            ];
            break;
    }
} catch (Exception $e) {
    $app->log('error', 'Data loading error: ' . $e->getMessage());
    $errorMessage = 'Error loading data. Please refresh the page.';
}

// Set page title
$pageTitles = [
    'dashboard' => 'Dashboard',
    'businesses' => 'Manage Businesses',
    'reviews' => 'Reviews',
    'analytics' => 'Analytics',
    'settings' => 'Settings',
    'qr-generator' => 'QR Code Generator',
];

$pageTitle = $pageTitles[$currentTab] ?? 'Admin Panel';

// Load the layout
include 'views/layout.php';
?>