<?php
/**
 * Elevadex Review Management System
 * Main Application Class
 * 
 * @version 2.0
 * @author Elevadex Development Team
 */

// Prevent direct access
if (!defined('ELEVADEX_APP')) {
    die('Direct access not allowed');
}

class ElevadexApp
{
    private static $instance = null;
    private $config = [];
    private $initialized = false;
    
    /**
     * Singleton pattern - get app instance
     */
    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct()
    {
        // Load configuration
        $this->loadConfiguration();
        
        // Initialize application
        $this->initialize();
    }
    
    /**
     * Initialize the application
     */
    private function initialize()
    {
        if ($this->initialized) {
            return;
        }
        
        try {
            // Set error reporting based on environment
            $this->setErrorReporting();
            
            // Set timezone
            date_default_timezone_set($this->config('app.timezone', 'UTC'));
            
            // Create required directories
            $this->createDirectories();
            
            // Initialize session
            $this->initializeSession();
            
            // Set up error and exception handlers
            $this->setErrorHandlers();
            
            // Load autoloader for classes
            $this->setupAutoloader();
            
            $this->initialized = true;
            
            // Log successful initialization
            $this->log('info', 'Application initialized successfully');
            
        } catch (Exception $e) {
            // Log critical initialization error
            error_log('Elevadex App Initialization Error: ' . $e->getMessage());
            die('Application initialization failed. Please check configuration and file permissions.');
        }
    }
    
    /**
     * Load application configuration
     */
    private function loadConfiguration()
    {
        $configFile = dirname(__DIR__) . '/config/app.php';
        
        if (!file_exists($configFile)) {
            die('Configuration file not found. Please ensure config/app.php exists.');
        }
        
        $this->config = require $configFile;
        
        // Override with environment-specific config if exists
        $envConfigFile = dirname(__DIR__) . '/config/environment.php';
        if (file_exists($envConfigFile)) {
            $envConfig = require $envConfigFile;
            $this->config = array_merge_recursive($this->config, $envConfig);
        }
    }
    
    /**
     * Get configuration value
     */
    public function config($key = null, $default = null)
    {
        if ($key === null) {
            return $this->config;
        }
        
        $keys = explode('.', $key);
        $value = $this->config;
        
        foreach ($keys as $k) {
            if (isset($value[$k])) {
                $value = $value[$k];
            } else {
                return $default;
            }
        }
        
        return $value;
    }
    
    /**
     * Set error reporting based on environment
     */
    private function setErrorReporting()
    {
        if ($this->config('app.debug', false)) {
            error_reporting(E_ALL);
            ini_set('display_errors', 1);
            ini_set('display_startup_errors', 1);
        } else {
            error_reporting(E_ALL & ~E_NOTICE & ~E_DEPRECATED);
            ini_set('display_errors', 0);
            ini_set('display_startup_errors', 0);
        }
        
        ini_set('log_errors', 1);
        ini_set('error_log', $this->config('paths.logs') . '/error.log');
    }
    
    /**
     * Create required directories
     */
    private function createDirectories()
    {
        $directories = [
            $this->config('paths.storage'),
            $this->config('paths.data'),
            $this->config('paths.uploads'),
            $this->config('paths.uploads') . '/logos',
            $this->config('paths.uploads') . '/temp',
            $this->config('paths.logs'),
            $this->config('paths.cache'),
            $this->config('paths.qr_codes'),
            $this->config('paths.pages'),
        ];
        
        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                if (!mkdir($dir, 0755, true)) {
                    throw new Exception("Failed to create directory: {$dir}");
                }
            }
        }
    }
    
    /**
     * Initialize session
     */
    private function initializeSession()
    {
        if (session_status() === PHP_SESSION_NONE) {
            // Configure session
            ini_set('session.name', $this->config('security.session_name'));
            ini_set('session.gc_maxlifetime', $this->config('security.session_lifetime'));
            ini_set('session.cookie_httponly', 1);
            
            if ($this->config('security.secure_cookies') && isset($_SERVER['HTTPS'])) {
                ini_set('session.cookie_secure', 1);
            }
            
            session_start();
        }
    }
    
    /**
     * Set up error and exception handlers
     */
    private function setErrorHandlers()
    {
        set_error_handler([$this, 'handleError']);
        set_exception_handler([$this, 'handleException']);
        register_shutdown_function([$this, 'handleShutdown']);
    }
    
    /**
     * Set up autoloader for classes
     */
    private function setupAutoloader()
    {
        spl_autoload_register(function ($className) {
            // Convert class name to file path
            $classFile = $this->config('paths.core') . '/' . str_replace('\\', '/', $className) . '.php';
            
            if (file_exists($classFile)) {
                require_once $classFile;
                return true;
            }
            
            return false;
        });
    }
    
    /**
     * Handle PHP errors
     */
    public function handleError($severity, $message, $file, $line)
    {
        if (!(error_reporting() & $severity)) {
            return false;
        }
        
        $errorInfo = [
            'type' => 'PHP Error',
            'severity' => $severity,
            'message' => $message,
            'file' => $file,
            'line' => $line,
            'timestamp' => date('Y-m-d H:i:s'),
            'url' => $_SERVER['REQUEST_URI'] ?? 'CLI',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
        ];
        
        $this->log('error', 'PHP Error: ' . $message, $errorInfo);
        
        if ($this->config('app.debug')) {
            echo "<div style='background:#f8d7da;color:#721c24;padding:15px;margin:10px;border-radius:5px;'>";
            echo "<strong>Error:</strong> {$message}<br>";
            echo "<strong>File:</strong> {$file}<br>";
            echo "<strong>Line:</strong> {$line}";
            echo "</div>";
        }
        
        return true;
    }
    
    /**
     * Handle uncaught exceptions
     */
    public function handleException($exception)
    {
        $exceptionInfo = [
            'type' => 'Uncaught Exception',
            'class' => get_class($exception),
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'timestamp' => date('Y-m-d H:i:s'),
            'url' => $_SERVER['REQUEST_URI'] ?? 'CLI',
        ];
        
        $this->log('error', 'Uncaught Exception: ' . $exception->getMessage(), $exceptionInfo);
        
        if ($this->config('app.debug')) {
            echo "<div style='background:#f8d7da;color:#721c24;padding:15px;margin:10px;border-radius:5px;'>";
            echo "<strong>Exception:</strong> " . $exception->getMessage() . "<br>";
            echo "<strong>File:</strong> " . $exception->getFile() . "<br>";
            echo "<strong>Line:</strong> " . $exception->getLine() . "<br>";
            echo "<pre>" . $exception->getTraceAsString() . "</pre>";
            echo "</div>";
        } else {
            echo "<h1>System Error</h1><p>An error occurred. Please try again later.</p>";
        }
    }
    
    /**
     * Handle script shutdown
     */
    public function handleShutdown()
    {
        $error = error_get_last();
        
        if ($error && in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_PARSE])) {
            $this->handleError($error['type'], $error['message'], $error['file'], $error['line']);
        }
    }
    
    /**
     * Log message to file
     */
    public function log($level, $message, $context = [])
    {
        if (!$this->config('logging.enabled')) {
            return;
        }
        
        $logFile = $this->config('paths.logs') . '/app.log';
        
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'level' => strtoupper($level),
            'message' => $message,
            'context' => $context,
            'memory' => memory_get_usage(true),
            'pid' => getmypid(),
        ];
        
        $logLine = json_encode($logEntry) . "\n";
        
        file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
        
        // Rotate log if too large
        if (filesize($logFile) > $this->config('logging.max_file_size')) {
            $this->rotateLog($logFile);
        }
    }
    
    /**
     * Rotate log file
     */
    private function rotateLog($logFile)
    {
        $backupFile = $logFile . '.' . date('Y-m-d-H-i-s');
        rename($logFile, $backupFile);
        
        // Clean old log files
        $this->cleanOldLogs();
    }
    
    /**
     * Clean old log files
     */
    private function cleanOldLogs()
    {
        $logDir = $this->config('paths.logs');
        $retentionDays = $this->config('logging.retention_days');
        $cutoffTime = time() - ($retentionDays * 24 * 60 * 60);
        
        $files = glob($logDir . '/*.log.*');
        
        foreach ($files as $file) {
            if (filemtime($file) < $cutoffTime) {
                unlink($file);
            }
        }
    }
    
    /**
     * Get application version
     */
    public function getVersion()
    {
        return $this->config('app.version');
    }
    
    /**
     * Check if application is in debug mode
     */
    public function isDebug()
    {
        return $this->config('app.debug', false);
    }
    
    /**
     * Check if application is in maintenance mode
     */
    public function isMaintenanceMode()
    {
        return $this->config('maintenance.mode', false);
    }
    
    /**
     * Get application environment
     */
    public function getEnvironment()
    {
        return $this->config('app.environment', 'production');
    }
}

// Initialize application
define('ELEVADEX_APP', true);
return ElevadexApp::getInstance();
?>