<?php
/**
 * Elevadex Review Management System
 * Business Data Model
 * 
 * @version 2.0
 * @author Elevadex Development Team
 */

// Prevent direct access
if (!defined('ELEVADEX_APP')) {
    die('Direct access not allowed');
}

class BusinessModel
{
    private $db;
    private $app;
    private $table = 'businesses';
    
    public function __construct($database)
    {
        $this->db = $database;
        $this->app = ElevadexApp::getInstance();
    }
    
    /**
     * Get all businesses
     */
    public function getAll()
    {
        try {
            $businesses = $this->db->read($this->table);
            
            // Sort by name
            usort($businesses, function($a, $b) {
                return strcmp($a['name'] ?? '', $b['name'] ?? '');
            });
            
            return $businesses;
            
        } catch (Exception $e) {
            $this->app->log('error', 'Error fetching businesses: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get business by ID
     */
    public function getById($id)
    {
        try {
            return $this->db->findById($this->table, $id);
        } catch (Exception $e) {
            $this->app->log('error', "Error fetching business {$id}: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get business by subdomain
     */
    public function getBySubdomain($subdomain)
    {
        try {
            $businesses = $this->db->find($this->table, ['subdomain' => $subdomain]);
            return !empty($businesses) ? $businesses[0] : null;
        } catch (Exception $e) {
            $this->app->log('error', "Error fetching business by subdomain {$subdomain}: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Create new business
     */
    public function create($data)
    {
        try {
            // Validate required fields
            $validation = $this->validateBusinessData($data);
            if (!$validation['valid']) {
                return [
                    'success' => false,
                    'message' => $validation['message'],
                    'errors' => $validation['errors'] ?? []
                ];
            }
            
            // Check if subdomain is already taken
            if ($this->isSubdomainTaken($data['subdomain'])) {
                return [
                    'success' => false,
                    'message' => 'Subdomain is already taken. Please choose a different one.',
                    'errors' => ['subdomain' => 'Already taken']
                ];
            }
            
            // Prepare business data
            $businessData = [
                'name' => trim($data['business_name']),
                'category' => trim($data['business_category']),
                'subdomain' => strtolower(trim($data['subdomain'])),
                'contact_email' => trim($data['contact_email'] ?? ''),
                'contact_phone' => trim($data['contact_phone'] ?? ''),
                'description' => trim($data['description'] ?? ''),
                'address' => trim($data['address'] ?? ''),
                'logo_path' => $data['logo_path'] ?? null,
                'gis_review_link' => trim($data['gis_review_link']),
                'google_review_link' => trim($data['google_review_link'] ?? ''),
                'yandex_review_link' => trim($data['yandex_review_link'] ?? ''),
                'status' => 'active',
                'pages_generated' => false,
                'qr_codes' => [],
                'current_qr' => null,
                'review_count' => 0,
                'external_review_count' => 0,
                'total_scans' => 0,
                'conversion_rate' => 0,
                'settings' => [
                    'positive_threshold' => $this->app->config('reviews.positive_threshold', 4),
                    'auto_generate_pages' => $this->app->config('business.auto_generate_pages', true),
                    'auto_generate_qr' => $this->app->config('business.auto_generate_qr', true),
                    'email_notifications' => true,
                    'public_display' => true
                ]
            ];
            
            // Insert business
            $business = $this->db->insert($this->table, $businessData);
            
            // Generate pages if auto-generation is enabled
            if ($business['settings']['auto_generate_pages']) {
                $this->generateBusinessPages($business);
            }
            
            // Generate QR code if auto-generation is enabled
            if ($business['settings']['auto_generate_qr']) {
                $this->generateQRCode($business['id']);
            }
            
            $this->app->log('info', "Business created: {$business['name']}", [
                'business_id' => $business['id'],
                'subdomain' => $business['subdomain']
            ]);
            
            return [
                'success' => true,
                'message' => "Business '{$business['name']}' created successfully!",
                'business' => $business
            ];
            
        } catch (Exception $e) {
            $this->app->log('error', 'Error creating business: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to create business. Please try again.'
            ];
        }
    }
    
    /**
     * Update existing business
     */
    public function update($id, $data)
    {
        try {
            // Get existing business
            $existing = $this->getById($id);
            if (!$existing) {
                return [
                    'success' => false,
                    'message' => 'Business not found.'
                ];
            }
            
            // Validate data
            $validation = $this->validateBusinessData($data, $id);
            if (!$validation['valid']) {
                return [
                    'success' => false,
                    'message' => $validation['message'],
                    'errors' => $validation['errors'] ?? []
                ];
            }
            
            // Check subdomain if changed
            if ($data['subdomain'] !== $existing['subdomain'] && 
                $this->isSubdomainTaken($data['subdomain'], $id)) {
                return [
                    'success' => false,
                    'message' => 'Subdomain is already taken.',
                    'errors' => ['subdomain' => 'Already taken']
                ];
            }
            
            // Prepare update data
            $updateData = [
                'name' => trim($data['business_name']),
                'category' => trim($data['business_category']),
                'subdomain' => strtolower(trim($data['subdomain'])),
                'contact_email' => trim($data['contact_email'] ?? ''),
                'contact_phone' => trim($data['contact_phone'] ?? ''),
                'description' => trim($data['description'] ?? ''),
                'address' => trim($data['address'] ?? ''),
                'gis_review_link' => trim($data['gis_review_link']),
                'google_review_link' => trim($data['google_review_link'] ?? ''),
                'yandex_review_link' => trim($data['yandex_review_link'] ?? ''),
            ];
            
            // Update logo if provided
            if (isset($data['logo_path']) && $data['logo_path']) {
                $updateData['logo_path'] = $data['logo_path'];
            }
            
            // Update business
            $business = $this->db->update($this->table, $id, $updateData);
            
            // Regenerate pages if subdomain changed
            if ($data['subdomain'] !== $existing['subdomain']) {
                $this->generateBusinessPages($business);
                $this->cleanupOldPages($existing['subdomain']);
            }
            
            $this->app->log('info', "Business updated: {$business['name']}", [
                'business_id' => $id,
                'changes' => array_keys($updateData)
            ]);
            
            return [
                'success' => true,
                'message' => "Business '{$business['name']}' updated successfully!",
                'business' => $business
            ];
            
        } catch (Exception $e) {
            $this->app->log('error', "Error updating business {$id}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to update business. Please try again.'
            ];
        }
    }
    
    /**
     * Delete business
     */
    public function delete($id)
    {
        try {
            $business = $this->getById($id);
            if (!$business) {
                return [
                    'success' => false,
                    'message' => 'Business not found.'
                ];
            }
            
            // Clean up associated files
            $this->cleanupBusinessFiles($business);
            
            // Delete from database
            $this->db->delete($this->table, $id);
            
            $this->app->log('info', "Business deleted: {$business['name']}", [
                'business_id' => $id,
                'subdomain' => $business['subdomain']
            ]);
            
            return [
                'success' => true,
                'message' => "Business '{$business['name']}' deleted successfully!"
            ];
            
        } catch (Exception $e) {
            $this->app->log('error', "Error deleting business {$id}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to delete business. Please try again.'
            ];
        }
    }
    
    /**
     * Toggle business status
     */
    public function toggleStatus($id)
    {
        try {
            $business = $this->getById($id);
            if (!$business) {
                return [
                    'success' => false,
                    'message' => 'Business not found.'
                ];
            }
            
            $newStatus = ($business['status'] === 'active') ? 'inactive' : 'active';
            
            $this->db->update($this->table, $id, ['status' => $newStatus]);
            
            $statusText = ($newStatus === 'active') ? 'activated' : 'deactivated';
            
            $this->app->log('info', "Business {$statusText}: {$business['name']}", [
                'business_id' => $id,
                'new_status' => $newStatus
            ]);
            
            return [
                'success' => true,
                'message' => "Business {$statusText} successfully!"
            ];
            
        } catch (Exception $e) {
            $this->app->log('error', "Error toggling business status {$id}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to update business status.'
            ];
        }
    }
    
    /**
     * Get business statistics
     */
    public function getStatistics($id = null)
    {
        try {
            if ($id) {
                // Statistics for specific business
                $business = $this->getById($id);
                if (!$business) {
                    return null;
                }
                
                return [
                    'total_reviews' => $business['review_count'] + $business['external_review_count'],
                    'internal_reviews' => $business['review_count'],
                    'external_reviews' => $business['external_review_count'],
                    'total_scans' => $business['total_scans'],
                    'conversion_rate' => $business['conversion_rate'],
                    'status' => $business['status'],
                    'pages_generated' => $business['pages_generated'],
                    'qr_codes_count' => count($business['qr_codes'])
                ];
            } else {
                // Overall statistics
                $businesses = $this->getAll();
                
                $stats = [
                    'total_businesses' => count($businesses),
                    'active_businesses' => 0,
                    'total_reviews' => 0,
                    'total_scans' => 0,
                    'pages_generated' => 0
                ];
                
                foreach ($businesses as $business) {
                    if ($business['status'] === 'active') {
                        $stats['active_businesses']++;
                    }
                    
                    $stats['total_reviews'] += $business['review_count'] + $business['external_review_count'];
                    $stats['total_scans'] += $business['total_scans'];
                    
                    if ($business['pages_generated']) {
                        $stats['pages_generated']++;
                    }
                }
                
                return $stats;
            }
            
        } catch (Exception $e) {
            $this->app->log('error', 'Error getting business statistics: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get business count
     */
    public function count()
    {
        try {
            return $this->db->count($this->table);
        } catch (Exception $e) {
            $this->app->log('error', 'Error counting businesses: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Search businesses
     */
    public function search($query)
    {
        try {
            $businesses = $this->getAll();
            $results = [];
            
            $query = strtolower(trim($query));
            
            foreach ($businesses as $business) {
                $searchText = strtolower(
                    $business['name'] . ' ' . 
                    $business['category'] . ' ' . 
                    $business['subdomain'] . ' ' . 
                    ($business['description'] ?? '')
                );
                
                if (strpos($searchText, $query) !== false) {
                    $results[] = $business;
                }
            }
            
            return $results;
            
        } catch (Exception $e) {
            $this->app->log('error', 'Error searching businesses: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Validate business data
     */
    private function validateBusinessData($data, $excludeId = null)
    {
        $errors = [];
        $requiredFields = ['business_name', 'business_category', 'subdomain', 'gis_review_link'];
        
        // Check required fields
        foreach ($requiredFields as $field) {
            if (empty($data[$field])) {
                $errors[$field] = 'This field is required';
            }
        }
        
        // Validate subdomain format
        if (!empty($data['subdomain'])) {
            $subdomain = strtolower(trim($data['subdomain']));
            
            if (!preg_match($this->app->config('business.subdomain_pattern'), $subdomain)) {
                $errors['subdomain'] = 'Only lowercase letters, numbers, and hyphens allowed';
            }
            
            if (strlen($subdomain) < $this->app->config('business.subdomain_min_length')) {
                $errors['subdomain'] = 'Minimum ' . $this->app->config('business.subdomain_min_length') . ' characters required';
            }
            
            if (strlen($subdomain) > $this->app->config('business.subdomain_max_length')) {
                $errors['subdomain'] = 'Maximum ' . $this->app->config('business.subdomain_max_length') . ' characters allowed';
            }
        }
        
        // Validate email if provided
        if (!empty($data['contact_email']) && !filter_var($data['contact_email'], FILTER_VALIDATE_EMAIL)) {
            $errors['contact_email'] = 'Invalid email address';
        }
        
        // Validate URLs
        $urlFields = ['gis_review_link', 'google_review_link', 'yandex_review_link'];
        foreach ($urlFields as $field) {
            if (!empty($data[$field]) && !filter_var($data[$field], FILTER_VALIDATE_URL)) {
                $errors[$field] = 'Invalid URL format';
            }
        }
        
        if (!empty($errors)) {
            return [
                'valid' => false,
                'message' => 'Please fix the validation errors below',
                'errors' => $errors
            ];
        }
        
        return ['valid' => true, 'message' => 'Valid'];
    }
    
    /**
     * Check if subdomain is taken
     */
    private function isSubdomainTaken($subdomain, $excludeId = null)
    {
        try {
            $businesses = $this->db->find($this->table, ['subdomain' => $subdomain]);
            
            if ($excludeId) {
                $businesses = array_filter($businesses, function($business) use ($excludeId) {
                    return $business['id'] != $excludeId;
                });
            }
            
            return !empty($businesses);
            
        } catch (Exception $e) {
            $this->app->log('error', 'Error checking subdomain: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Generate business pages (placeholder - implement in PageGenerator)
     */
    private function generateBusinessPages($business)
    {
        try {
            // This will be implemented in PageGenerator class
            require_once '../core/generators/PageGenerator.php';
            $pageGenerator = new PageGenerator($this->app);
            return $pageGenerator->generateBusinessPages($business);
        } catch (Exception $e) {
            $this->app->log('warning', 'Failed to generate pages for business: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Generate QR code (placeholder - implement in QRGenerator)
     */
    private function generateQRCode($businessId)
    {
        try {
            // This will be implemented in QRGenerator class
            require_once '../core/generators/QRGenerator.php';
            $qrGenerator = new QRGenerator($this->app);
            return $qrGenerator->generateForBusiness($businessId);
        } catch (Exception $e) {
            $this->app->log('warning', 'Failed to generate QR code for business: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Cleanup business files
     */
    private function cleanupBusinessFiles($business)
    {
        try {
            // Remove logo file
            if (!empty($business['logo_path']) && file_exists($business['logo_path'])) {
                unlink($business['logo_path']);
            }
            
            // Remove QR codes
            foreach ($business['qr_codes'] as $qrCode) {
                if (file_exists($qrCode['file_path'])) {
                    unlink($qrCode['file_path']);
                }
            }
            
            // Remove generated pages
            $this->cleanupOldPages($business['subdomain']);
            
        } catch (Exception $e) {
            $this->app->log('warning', 'Error cleaning up business files: ' . $e->getMessage());
        }
    }
    
    /**
     * Cleanup old pages
     */
    private function cleanupOldPages($subdomain)
    {
        try {
            $pagesDir = $this->app->config('paths.pages') . '/' . $subdomain;
            
            if (is_dir($pagesDir)) {
                $files = glob($pagesDir . '/*');
                foreach ($files as $file) {
                    if (is_file($file)) {
                        unlink($file);
                    }
                }
                rmdir($pagesDir);
            }
            
        } catch (Exception $e) {
            $this->app->log('warning', 'Error cleaning up old pages: ' . $e->getMessage());
        }
    }
}
?>