<?php
/**
 * Elevadex Review Management System
 * JSON Database Handler
 * 
 * @version 2.0
 * @author Elevadex Development Team
 */

// Prevent direct access
if (!defined('ELEVADEX_APP')) {
    die('Direct access not allowed');
}

class JsonDatabase
{
    private $app;
    private $dataPath;
    private $cache = [];
    private $cacheEnabled = true;
    
    public function __construct($app)
    {
        $this->app = $app;
        $this->dataPath = $app->config('paths.data');
        $this->cacheEnabled = $app->config('performance.cache_enabled', true);
        
        // Initialize data files
        $this->initializeDataFiles();
    }
    
    /**
     * Initialize required data files
     */
    private function initializeDataFiles()
    {
        $dataFiles = [
            'businesses.json' => [],
            'reviews.json' => [],
            'external_reviews.json' => [],
            'analytics.json' => [
                'last_updated' => date('Y-m-d H:i:s'),
                'daily_stats' => [],
                'monthly_stats' => [],
                'platform_stats' => []
            ],
            'settings.json' => [
                'version' => $this->app->config('app.version'),
                'installed_at' => date('Y-m-d H:i:s'),
                'last_backup' => null,
                'maintenance_mode' => false
            ],
            'failed_attempts.json' => [],
            'lockouts.json' => [],
            'remember_tokens.json' => []
        ];
        
        foreach ($dataFiles as $filename => $defaultData) {
            $filepath = $this->dataPath . '/' . $filename;
            
            if (!file_exists($filepath)) {
                if (!$this->writeJsonFile($filepath, $defaultData)) {
                    throw new Exception("Failed to create data file: {$filename}");
                }
                $this->app->log('info', "Created data file: {$filename}");
            }
        }
    }
    
    /**
     * Read data from JSON file
     */
    public function read($table, $useCache = true)
    {
        try {
            // Check cache first
            if ($useCache && $this->cacheEnabled && isset($this->cache[$table])) {
                return $this->cache[$table];
            }
            
            $filepath = $this->dataPath . '/' . $table . '.json';
            
            if (!file_exists($filepath)) {
                throw new Exception("Data file not found: {$table}.json");
            }
            
            $data = $this->readJsonFile($filepath);
            
            // Cache the data
            if ($this->cacheEnabled) {
                $this->cache[$table] = $data;
            }
            
            return $data;
            
        } catch (Exception $e) {
            $this->app->log('error', "Database read error for {$table}: " . $e->getMessage());
            throw new Exception("Failed to read data from {$table}");
        }
    }
    
    /**
     * Write data to JSON file
     */
    public function write($table, $data, $backup = true)
    {
        try {
            $filepath = $this->dataPath . '/' . $table . '.json';
            
            // Create backup if requested
            if ($backup && file_exists($filepath)) {
                $this->createBackup($table);
            }
            
            // Validate data structure
            if (!is_array($data)) {
                throw new Exception("Data must be an array");
            }
            
            // Write the data
            if (!$this->writeJsonFile($filepath, $data)) {
                throw new Exception("Failed to write to file");
            }
            
            // Update cache
            if ($this->cacheEnabled) {
                $this->cache[$table] = $data;
            }
            
            // Log the operation
            $this->app->log('info', "Data written to {$table}", [
                'record_count' => count($data),
                'file_size' => filesize($filepath)
            ]);
            
            return true;
            
        } catch (Exception $e) {
            $this->app->log('error', "Database write error for {$table}: " . $e->getMessage());
            throw new Exception("Failed to write data to {$table}");
        }
    }
    
    /**
     * Read JSON file with error handling
     */
    private function readJsonFile($filepath)
    {
        // Check file permissions
        if (!is_readable($filepath)) {
            throw new Exception("File is not readable: {$filepath}");
        }
        
        // Read file with file locking
        $handle = fopen($filepath, 'r');
        if (!$handle) {
            throw new Exception("Cannot open file: {$filepath}");
        }
        
        if (!flock($handle, LOCK_SH)) {
            fclose($handle);
            throw new Exception("Cannot lock file for reading: {$filepath}");
        }
        
        $content = stream_get_contents($handle);
        
        flock($handle, LOCK_UN);
        fclose($handle);
        
        if ($content === false) {
            throw new Exception("Cannot read file content: {$filepath}");
        }
        
        // Handle empty files
        if (trim($content) === '') {
            return [];
        }
        
        // Decode JSON
        $data = json_decode($content, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("JSON decode error: " . json_last_error_msg());
        }
        
        return $data ?? [];
    }
    
    /**
     * Write JSON file with error handling
     */
    private function writeJsonFile($filepath, $data)
    {
        // Ensure directory exists
        $directory = dirname($filepath);
        if (!is_dir($directory) && !mkdir($directory, 0755, true)) {
            throw new Exception("Cannot create directory: {$directory}");
        }
        
        // Encode JSON with pretty printing
        $jsonContent = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        
        if ($jsonContent === false) {
            throw new Exception("JSON encode error: " . json_last_error_msg());
        }
        
        // Write to temporary file first
        $tempFile = $filepath . '.tmp';
        
        $handle = fopen($tempFile, 'w');
        if (!$handle) {
            throw new Exception("Cannot create temporary file: {$tempFile}");
        }
        
        if (!flock($handle, LOCK_EX)) {
            fclose($handle);
            unlink($tempFile);
            throw new Exception("Cannot lock temporary file: {$tempFile}");
        }
        
        $bytesWritten = fwrite($handle, $jsonContent);
        
        if ($bytesWritten === false || $bytesWritten !== strlen($jsonContent)) {
            flock($handle, LOCK_UN);
            fclose($handle);
            unlink($tempFile);
            throw new Exception("Failed to write complete data to temporary file");
        }
        
        // Flush and sync to disk
        fflush($handle);
        if (function_exists('fsync')) {
            fsync($handle);
        }
        
        flock($handle, LOCK_UN);
        fclose($handle);
        
        // Atomic move to final location
        if (!rename($tempFile, $filepath)) {
            unlink($tempFile);
            throw new Exception("Cannot move temporary file to final location");
        }
        
        return true;
    }
    
    /**
     * Create backup of data file
     */
    private function createBackup($table)
    {
        try {
            $sourceFile = $this->dataPath . '/' . $table . '.json';
            $backupDir = $this->dataPath . '/backups';
            
            // Create backup directory if it doesn't exist
            if (!is_dir($backupDir) && !mkdir($backupDir, 0755, true)) {
                throw new Exception("Cannot create backup directory");
            }
            
            $timestamp = date('Y-m-d_H-i-s');
            $backupFile = $backupDir . '/' . $table . '_' . $timestamp . '.json';
            
            if (!copy($sourceFile, $backupFile)) {
                throw new Exception("Cannot create backup file");
            }
            
            // Cleanup old backups (keep last 10)
            $this->cleanupBackups($table, 10);
            
            $this->app->log('info', "Backup created for {$table}", [
                'backup_file' => basename($backupFile)
            ]);
            
        } catch (Exception $e) {
            $this->app->log('warning', "Backup creation failed for {$table}: " . $e->getMessage());
            // Don't throw exception for backup failures to avoid breaking the main operation
        }
    }
    
    /**
     * Cleanup old backup files
     */
    private function cleanupBackups($table, $keepCount = 10)
    {
        $backupDir = $this->dataPath . '/backups';
        $pattern = $backupDir . '/' . $table . '_*.json';
        $backupFiles = glob($pattern);
        
        if (count($backupFiles) > $keepCount) {
            // Sort by modification time (oldest first)
            usort($backupFiles, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            
            // Remove oldest files
            $filesToRemove = array_slice($backupFiles, 0, count($backupFiles) - $keepCount);
            
            foreach ($filesToRemove as $file) {
                if (unlink($file)) {
                    $this->app->log('info', "Removed old backup: " . basename($file));
                }
            }
        }
    }
    
    /**
     * Insert new record
     */
    public function insert($table, $record)
    {
        $data = $this->read($table);
        
        // Generate ID if not provided
        if (!isset($record['id'])) {
            $record['id'] = $this->generateId($data);
        }
        
        // Add timestamps
        $record['created_at'] = date('Y-m-d H:i:s');
        $record['updated_at'] = date('Y-m-d H:i:s');
        
        $data[] = $record;
        
        $this->write($table, $data);
        
        return $record;
    }
    
    /**
     * Update existing record
     */
    public function update($table, $id, $updates)
    {
        $data = $this->read($table);
        $updated = false;
        
        foreach ($data as &$record) {
            if ($record['id'] == $id) {
                $record = array_merge($record, $updates);
                $record['updated_at'] = date('Y-m-d H:i:s');
                $updated = true;
                break;
            }
        }
        
        if (!$updated) {
            throw new Exception("Record not found with ID: {$id}");
        }
        
        $this->write($table, $data);
        
        return $record;
    }
    
    /**
     * Delete record
     */
    public function delete($table, $id)
    {
        $data = $this->read($table);
        $originalCount = count($data);
        
        $data = array_filter($data, function($record) use ($id) {
            return $record['id'] != $id;
        });
        
        // Re-index array
        $data = array_values($data);
        
        if (count($data) === $originalCount) {
            throw new Exception("Record not found with ID: {$id}");
        }
        
        $this->write($table, $data);
        
        return true;
    }
    
    /**
     * Find record by ID
     */
    public function findById($table, $id)
    {
        $data = $this->read($table);
        
        foreach ($data as $record) {
            if ($record['id'] == $id) {
                return $record;
            }
        }
        
        return null;
    }
    
    /**
     * Find records by criteria
     */
    public function find($table, $criteria = [], $limit = null, $offset = 0)
    {
        $data = $this->read($table);
        $results = [];
        
        foreach ($data as $record) {
            $match = true;
            
            foreach ($criteria as $field => $value) {
                if (!isset($record[$field]) || $record[$field] != $value) {
                    $match = false;
                    break;
                }
            }
            
            if ($match) {
                $results[] = $record;
            }
        }
        
        // Apply offset and limit
        if ($offset > 0) {
            $results = array_slice($results, $offset);
        }
        
        if ($limit !== null && $limit > 0) {
            $results = array_slice($results, 0, $limit);
        }
        
        return $results;
    }
    
    /**
     * Count records
     */
    public function count($table, $criteria = [])
    {
        if (empty($criteria)) {
            $data = $this->read($table);
            return count($data);
        }
        
        return count($this->find($table, $criteria));
    }
    
    /**
     * Generate unique ID
     */
    private function generateId($data)
    {
        if (empty($data)) {
            return 1;
        }
        
        $maxId = 0;
        foreach ($data as $record) {
            if (isset($record['id']) && $record['id'] > $maxId) {
                $maxId = $record['id'];
            }
        }
        
        return $maxId + 1;
    }
    
    /**
     * Clear cache for specific table or all tables
     */
    public function clearCache($table = null)
    {
        if ($table === null) {
            $this->cache = [];
        } else {
            unset($this->cache[$table]);
        }
    }
    
    /**
     * Get database statistics
     */
    public function getStats()
    {
        $stats = [
            'tables' => [],
            'total_records' => 0,
            'total_size' => 0,
            'cache_status' => $this->cacheEnabled,
            'cached_tables' => array_keys($this->cache)
        ];
        
        $dataFiles = glob($this->dataPath . '/*.json');
        
        foreach ($dataFiles as $file) {
            $tableName = basename($file, '.json');
            
            if (in_array($tableName, ['failed_attempts', 'lockouts', 'remember_tokens'])) {
                continue; // Skip system files
            }
            
            $size = filesize($file);
            $records = count($this->read($tableName, false)); // Don't use cache for stats
            
            $stats['tables'][$tableName] = [
                'records' => $records,
                'size' => $size,
                'last_modified' => date('Y-m-d H:i:s', filemtime($file))
            ];
            
            $stats['total_records'] += $records;
            $stats['total_size'] += $size;
        }
        
        return $stats;
    }
    
    /**
     * Optimize database (cleanup and defragment)
     */
    public function optimize()
    {
        $optimized = [];
        
        try {
            // Clear cache
            $this->clearCache();
            
            // Rewrite all data files to clean up formatting
            $dataFiles = glob($this->dataPath . '/*.json');
            
            foreach ($dataFiles as $file) {
                $tableName = basename($file, '.json');
                
                if (in_array($tableName, ['failed_attempts', 'lockouts', 'remember_tokens'])) {
                    continue; // Skip system files
                }
                
                $data = $this->read($tableName, false);
                $this->write($tableName, $data, false); // Don't create backup during optimization
                
                $optimized[] = $tableName;
            }
            
            // Cleanup old backup files
            $backupDir = $this->dataPath . '/backups';
            if (is_dir($backupDir)) {
                $backupFiles = glob($backupDir . '/*.json');
                $cutoff = time() - (30 * 24 * 60 * 60); // 30 days
                
                foreach ($backupFiles as $file) {
                    if (filemtime($file) < $cutoff) {
                        unlink($file);
                    }
                }
            }
            
            $this->app->log('info', 'Database optimization completed', [
                'optimized_tables' => $optimized
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Export data to various formats
     */
    public function export($table, $format = 'json')
    {
        $data = $this->read($table);
        
        switch (strtolower($format)) {
            case 'json':
                return json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                
            case 'csv':
                if (empty($data)) {
                    return '';
                }
                
                $output = fopen('php://temp', 'r+');
                
                // Write header
                fputcsv($output, array_keys($data[0]));
                
                // Write data
                foreach ($data as $record) {
                    fputcsv($output, $record);
                }
                
                rewind($output);
                $csv = stream_get_contents($output);
                fclose($output);
                
                return $csv;
                
            case 'xml':
                $xml = new SimpleXMLElement('<data/>');
                
                foreach ($data as $record) {
                    $item = $xml->addChild('record');
                    foreach ($record as $key => $value) {
                        $item->addChild($key, htmlspecialchars($value));
                    }
                }
                
                return $xml->asXML();
                
            default:
                throw new Exception("Unsupported export format: {$format}");
        }
    }
    
    /**
     * Import data from various formats
     */
    public function import($table, $data, $format = 'json', $mode = 'append')
    {
        switch (strtolower($format)) {
            case 'json':
                $importData = json_decode($data, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception("Invalid JSON data: " . json_last_error_msg());
                }
                break;
                
            case 'csv':
                $importData = [];
                $lines = str_getcsv($data, "\n");
                $headers = str_getcsv(array_shift($lines));
                
                foreach ($lines as $line) {
                    $values = str_getcsv($line);
                    if (count($values) === count($headers)) {
                        $importData[] = array_combine($headers, $values);
                    }
                }
                break;
                
            default:
                throw new Exception("Unsupported import format: {$format}");
        }
        
        if (!is_array($importData)) {
            throw new Exception("Import data must be an array");
        }
        
        // Handle import mode
        switch ($mode) {
            case 'replace':
                $this->write($table, $importData);
                break;
                
            case 'append':
                $existingData = $this->read($table);
                $mergedData = array_merge($existingData, $importData);
                $this->write($table, $mergedData);
                break;
                
            case 'update':
                $existingData = $this->read($table);
                
                foreach ($importData as $importRecord) {
                    if (isset($importRecord['id'])) {
                        $found = false;
                        foreach ($existingData as &$existingRecord) {
                            if ($existingRecord['id'] == $importRecord['id']) {
                                $existingRecord = array_merge($existingRecord, $importRecord);
                                $existingRecord['updated_at'] = date('Y-m-d H:i:s');
                                $found = true;
                                break;
                            }
                        }
                        
                        if (!$found) {
                            $existingData[] = $importRecord;
                        }
                    } else {
                        $existingData[] = $importRecord;
                    }
                }
                
                $this->write($table, $existingData);
                break;
                
            default:
                throw new Exception("Invalid import mode: {$mode}");
        }
        
        return count($importData);
    }
}
?> => true,
                'optimized_tables' => $optimized
            ];
            
        } catch (Exception $e) {
            $this->app->log('error', 'Database optimization failed: ' . $e->getMessage());
            
            return [
                'success'