<?php
/**
 * Elevadex Review Management System
 * Installation Script
 * 
 * @version 2.0
 * @author Elevadex Development Team
 */

// Set error reporting for installation
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Installation status
$installationSteps = [
    'check_requirements' => ['name' => 'System Requirements Check', 'status' => 'pending'],
    'create_structure' => ['name' => 'Create Directory Structure', 'status' => 'pending'],
    'set_permissions' => ['name' => 'Set File Permissions', 'status' => 'pending'],
    'initialize_config' => ['name' => 'Initialize Configuration', 'status' => 'pending'],
    'create_htaccess' => ['name' => 'Create .htaccess Rules', 'status' => 'pending'],
    'setup_database' => ['name' => 'Setup JSON Database', 'status' => 'pending'],
    'create_admin_assets' => ['name' => 'Create Admin Assets', 'status' => 'pending'],
    'test_system' => ['name' => 'System Test', 'status' => 'pending'],
];

$errors = [];
$warnings = [];
$success = [];

// Handle installation request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'install') {
    $adminPassword = $_POST['admin_password'] ?? '';
    $siteName = $_POST['site_name'] ?? 'Elevadex Review Management System';
    $adminEmail = $_POST['admin_email'] ?? 'admin@elevadex.com';
    $siteUrl = $_POST['site_url'] ?? 'https://elevadex.com';
    
    if (strlen($adminPassword) < 6) {
        $errors[] = 'Admin password must be at least 6 characters long';
    } else {
        // Run installation
        runInstallation($adminPassword, $siteName, $adminEmail, $siteUrl);
    }
}

function runInstallation($adminPassword, $siteName, $adminEmail, $siteUrl) {
    global $installationSteps, $errors, $warnings, $success;
    
    // Step 1: Check Requirements
    $installationSteps['check_requirements']['status'] = 'running';
    if (checkSystemRequirements()) {
        $installationSteps['check_requirements']['status'] = 'completed';
        $success[] = 'System requirements check passed';
    } else {
        $installationSteps['check_requirements']['status'] = 'failed';
        return false;
    }
    
    // Step 2: Create Directory Structure
    $installationSteps['create_structure']['status'] = 'running';
    if (createDirectoryStructure()) {
        $installationSteps['create_structure']['status'] = 'completed';
        $success[] = 'Directory structure created successfully';
    } else {
        $installationSteps['create_structure']['status'] = 'failed';
        return false;
    }
    
    // Step 3: Set Permissions
    $installationSteps['set_permissions']['status'] = 'running';
    if (setFilePermissions()) {
        $installationSteps['set_permissions']['status'] = 'completed';
        $success[] = 'File permissions set correctly';
    } else {
        $installationSteps['set_permissions']['status'] = 'failed';
        return false;
    }
    
    // Step 4: Initialize Configuration
    $installationSteps['initialize_config']['status'] = 'running';
    if (initializeConfiguration($adminPassword, $siteName, $adminEmail, $siteUrl)) {
        $installationSteps['initialize_config']['status'] = 'completed';
        $success[] = 'Configuration files created';
    } else {
        $installationSteps['initialize_config']['status'] = 'failed';
        return false;
    }
    
    // Step 5: Create .htaccess
    $installationSteps['create_htaccess']['status'] = 'running';
    if (createHtaccessRules()) {
        $installationSteps['create_htaccess']['status'] = 'completed';
        $success[] = 'URL rewrite rules created';
    } else {
        $installationSteps['create_htaccess']['status'] = 'failed';
        return false;
    }
    
    // Step 6: Setup Database
    $installationSteps['setup_database']['status'] = 'running';
    if (setupJsonDatabase()) {
        $installationSteps['setup_database']['status'] = 'completed';
        $success[] = 'JSON database initialized';
    } else {
        $installationSteps['setup_database']['status'] = 'failed';
        return false;
    }
    
    // Step 7: Create Admin Assets
    $installationSteps['create_admin_assets']['status'] = 'running';
    if (createAdminAssets()) {
        $installationSteps['create_admin_assets']['status'] = 'completed';
        $success[] = 'Admin assets created';
    } else {
        $installationSteps['create_admin_assets']['status'] = 'failed';
        return false;
    }
    
    // Step 8: Test System
    $installationSteps['test_system']['status'] = 'running';
    if (testSystem()) {
        $installationSteps['test_system']['status'] = 'completed';
        $success[] = 'System test passed - Installation completed successfully!';
        
        // Create installation complete flag
        file_put_contents('INSTALLATION_COMPLETE', date('Y-m-d H:i:s'));
        
        return true;
    } else {
        $installationSteps['test_system']['status'] = 'failed';
        return false;
    }
}

function checkSystemRequirements() {
    global $errors, $warnings;
    
    $requirements = [
        'php_version' => version_compare(PHP_VERSION, '7.4.0', '>='),
        'json_extension' => extension_loaded('json'),
        'gd_extension' => extension_loaded('gd'),
        'curl_extension' => extension_loaded('curl'),
        'fileinfo_extension' => extension_loaded('fileinfo'),
        'mbstring_extension' => extension_loaded('mbstring'),
        'write_permissions' => is_writable('.'),
        'memory_limit' => ini_get('memory_limit') >= '128M',
    ];
    
    $passed = true;
    
    if (!$requirements['php_version']) {
        $errors[] = 'PHP 7.4.0 or higher is required. Current version: ' . PHP_VERSION;
        $passed = false;
    }
    
    if (!$requirements['json_extension']) {
        $errors[] = 'JSON extension is required';
        $passed = false;
    }
    
    if (!$requirements['gd_extension']) {
        $warnings[] = 'GD extension is recommended for image processing';
    }
    
    if (!$requirements['curl_extension']) {
        $warnings[] = 'cURL extension is recommended for external API calls';
    }
    
    if (!$requirements['fileinfo_extension']) {
        $warnings[] = 'Fileinfo extension is recommended for file uploads';
    }
    
    if (!$requirements['mbstring_extension']) {
        $warnings[] = 'Mbstring extension is recommended for text processing';
    }
    
    if (!$requirements['write_permissions']) {
        $errors[] = 'Write permissions required in current directory';
        $passed = false;
    }
    
    if (!$requirements['memory_limit']) {
        $warnings[] = 'Memory limit of 128M or higher is recommended';
    }
    
    return $passed;
}

function createDirectoryStructure() {
    global $errors;
    
    $directories = [
        'config',
        'core',
        'core/auth',
        'core/database',
        'core/generators',
        'core/helpers',
        'admin',
        'admin/controllers',
        'admin/views',
        'admin/views/businesses',
        'admin/views/reviews',
        'admin/views/analytics',
        'admin/assets',
        'admin/assets/css',
        'admin/assets/js',
        'admin/assets/images',
        'api',
        'api/v1',
        'api/middleware',
        'public',
        'public/assets',
        'public/assets/css',
        'public/assets/js',
        'public/assets/images',
        'public/templates',
        'storage',
        'storage/data',
        'storage/data/backups',
        'storage/uploads',
        'storage/uploads/logos',
        'storage/uploads/temp',
        'storage/generated',
        'storage/generated/qr-codes',
        'storage/generated/pages',
        'storage/generated/cache',
        'storage/logs',
        'scripts'
    ];
    
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0755, true)) {
                $errors[] = "Failed to create directory: {$dir}";
                return false;
            }
        }
    }
    
    return true;
}

function setFilePermissions() {
    global $warnings;
    
    $permissions = [
        'storage' => 0777,
        'storage/data' => 0777,
        'storage/uploads' => 0777,
        'storage/logs' => 0777,
        'storage/generated' => 0777,
        'config' => 0755,
        'admin' => 0755,
        'core' => 0755,
    ];
    
    foreach ($permissions as $path => $permission) {
        if (is_dir($path)) {
            if (!chmod($path, $permission)) {
                $warnings[] = "Could not set permissions for: {$path}";
            }
        }
    }
    
    return true;
}

function initializeConfiguration($adminPassword, $siteName, $adminEmail, $siteUrl) {
    global $errors;
    
    try {
        // Create app.php config
        $appConfig = generateAppConfig($adminPassword, $siteName, $adminEmail, $siteUrl);
        if (!file_put_contents('config/app.php', $appConfig)) {
            $errors[] = 'Failed to create app configuration file';
            return false;
        }
        
        // Create environment.php config (optional overrides)
        $envConfig = generateEnvironmentConfig();
        if (!file_put_contents('config/environment.php', $envConfig)) {
            $errors[] = 'Failed to create environment configuration file';
            return false;
        }
        
        return true;
        
    } catch (Exception $e) {
        $errors[] = 'Configuration creation failed: ' . $e->getMessage();
        return false;
    }
}

function generateAppConfig($adminPassword, $siteName, $adminEmail, $siteUrl) {
    return '<?php
/**
 * Elevadex Review Management System
 * Application Configuration
 * Generated by installation script
 */

if (!defined(\'ELEVADEX_APP\')) {
    die(\'Direct access not allowed\');
}

return [
    // Application Settings
    \'app\' => [
        \'name\' => \'' . addslashes($siteName) . '\',
        \'version\' => \'2.0.0\',
        \'environment\' => \'production\',
        \'debug\' => false,
        \'timezone\' => \'Asia/Almaty\',
        \'locale\' => \'en\',
        \'url\' => \'' . addslashes($siteUrl) . '\',
        \'admin_url\' => \'' . addslashes($siteUrl) . '/admin\',
    ],

    // Security Settings
    \'security\' => [
        \'admin_password\' => \'' . addslashes($adminPassword) . '\',
        \'session_name\' => \'elevadex_admin_session\',
        \'session_lifetime\' => 7200,
        \'csrf_protection\' => true,
        \'rate_limiting\' => true,
        \'max_login_attempts\' => 5,
        \'lockout_duration\' => 900,
        \'secure_cookies\' => true,
    ],

    // File & Upload Settings
    \'uploads\' => [
        \'max_file_size\' => 5 * 1024 * 1024,
        \'allowed_types\' => [\'jpg\', \'jpeg\', \'png\', \'gif\', \'webp\'],
        \'logo_max_width\' => 500,
        \'logo_max_height\' => 500,
    ],

    // Paths
    \'paths\' => [
        \'root\' => dirname(__DIR__),
        \'storage\' => dirname(__DIR__) . \'/storage\',
        \'data\' => dirname(__DIR__) . \'/storage/data\',
        \'uploads\' => dirname(__DIR__) . \'/storage/uploads\',
        \'logs\' => dirname(__DIR__) . \'/storage/logs\',
        \'cache\' => dirname(__DIR__) . \'/storage/generated/cache\',
        \'qr_codes\' => dirname(__DIR__) . \'/storage/generated/qr-codes\',
        \'pages\' => dirname(__DIR__) . \'/storage/generated/pages\',
    ],

    // Review System
    \'reviews\' => [
        \'positive_threshold\' => 4,
        \'negative_threshold\' => 3,
        \'auto_email_notifications\' => true,
        \'admin_email\' => \'' . addslashes($adminEmail) . '\',
    ],

    // Business Settings
    \'business\' => [
        \'subdomain_min_length\' => 3,
        \'subdomain_max_length\' => 50,
        \'subdomain_pattern\' => \'/^[a-z0-9-]+$/\',
        \'auto_generate_pages\' => true,
        \'auto_generate_qr\' => true,
    ],

    // Logging
    \'logging\' => [
        \'enabled\' => true,
        \'level\' => \'info\',
        \'max_file_size\' => 10 * 1024 * 1024,
        \'retention_days\' => 30,
    ],

    // Performance
    \'performance\' => [
        \'cache_enabled\' => true,
        \'gzip_compression\' => true,
    ],
];
?>';
}

function generateEnvironmentConfig() {
    return '<?php
/**
 * Environment-specific configuration overrides
 * This file can be used to override settings for different environments
 */

if (!defined(\'ELEVADEX_APP\')) {
    die(\'Direct access not allowed\');
}

// Development environment overrides
if ($_SERVER[\'HTTP_HOST\'] === \'localhost\' || strpos($_SERVER[\'HTTP_HOST\'], \'dev.\') === 0) {
    return [
        \'app\' => [
            \'environment\' => \'development\',
            \'debug\' => true,
        ],
        \'logging\' => [
            \'level\' => \'debug\',
        ],
    ];
}

// Staging environment overrides
if (strpos($_SERVER[\'HTTP_HOST\'], \'staging.\') === 0) {
    return [
        \'app\' => [
            \'environment\' => \'staging\',
            \'debug\' => false,
        ],
    ];
}

// Production environment (default)
return [];
?>';
}

function createHtaccessRules() {
    global $errors;
    
    $htaccessContent = '# Elevadex Review Management System
# URL Rewrite Rules

RewriteEngine On

# Security: Block access to sensitive files
<FilesMatch "\\.(json|log|txt|md)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# Block access to config and storage directories
RedirectMatch 403 ^/(config|storage|core|scripts)/.*$

# Subdomain routing for business pages
RewriteCond %{HTTP_HOST} ^([^.]+)\\.elevadex\\.com$ [NC]
RewriteCond %1 !^(www|admin|api)$ [NC]
RewriteRule ^(.*)$ /public/index.php?subdomain=%1&path=$1 [L,QSA]

# Admin panel routing
RewriteRule ^admin/?(.*)$ /admin/index.php [L,QSA]

# API routing
RewriteRule ^api/(.*)$ /api/index.php [L,QSA]

# Main site routing
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ /public/index.php [L,QSA]

# Security headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>

# Cache control
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
</IfModule>
';

    if (!file_put_contents('.htaccess', $htaccessContent)) {
        $errors[] = 'Failed to create .htaccess file';
        return false;
    }
    
    return true;
}

function setupJsonDatabase() {
    global $errors;
    
    try {
        // Initialize basic data files
        $dataFiles = [
            'storage/data/businesses.json' => [],
            'storage/data/reviews.json' => [],
            'storage/data/external_reviews.json' => [],
            'storage/data/analytics.json' => [
                'last_updated' => date('Y-m-d H:i:s'),
                'daily_stats' => [],
                'monthly_stats' => [],
            ],
            'storage/data/settings.json' => [
                'version' => '2.0.0',
                'installed_at' => date('Y-m-d H:i:s'),
                'installation_id' => uniqid(),
            ],
        ];
        
        foreach ($dataFiles as $file => $data) {
            if (!file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT))) {
                $errors[] = "Failed to create data file: {$file}";
                return false;
            }
        }
        
        return true;
        
    } catch (Exception $e) {
        $errors[] = 'Database setup failed: ' . $e->getMessage();
        return false;
    }
}

function createAdminAssets() {
    global $errors;
    
    try {
        // Create basic admin CSS
        $adminCSS = generateAdminCSS();
        if (!file_put_contents('admin/assets/css/admin.css', $adminCSS)) {
            $errors[] = 'Failed to create admin CSS file';
            return false;
        }
        
        // Create basic admin JS
        $adminJS = generateAdminJS();
        if (!file_put_contents('admin/assets/js/admin.js', $adminJS)) {
            $errors[] = 'Failed to create admin JS file';
            return false;
        }
        
        return true;
        
    } catch (Exception $e) {
        $errors[] = 'Admin assets creation failed: ' . $e->getMessage();
        return false;
    }
}

function generateAdminCSS() {
    return '/* Elevadex Admin Panel Styles */
* {
    box-sizing: border-box;
}

body {
    margin: 0;
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    background: #f8f9fa;
    color: #333;
}

.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.btn {
    display: inline-block;
    padding: 10px 20px;
    background: #667eea;
    color: white;
    text-decoration: none;
    border-radius: 5px;
    border: none;
    cursor: pointer;
    transition: background 0.3s ease;
}

.btn:hover {
    background: #5a6fd8;
}

.btn-secondary {
    background: #6c757d;
}

.btn-secondary:hover {
    background: #5a6268;
}

.card {
    background: white;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.form-group {
    margin-bottom: 20px;
}

.form-label {
    display: block;
    font-weight: 600;
    margin-bottom: 5px;
}

.form-control {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 5px;
    font-size: 14px;
}

.form-control:focus {
    outline: none;
    border-color: #667eea;
}

.alert {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
}

.alert-success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.alert-error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.text-center {
    text-align: center;
}

.mt-3 {
    margin-top: 1rem;
}

.mb-3 {
    margin-bottom: 1rem;
}
';
}

function generateAdminJS() {
    return '// Elevadex Admin Panel JavaScript
document.addEventListener("DOMContentLoaded", function() {
    console.log("Elevadex Admin Panel loaded");
    
    // Form validation
    const forms = document.querySelectorAll("form");
    forms.forEach(form => {
        form.addEventListener("submit", function(e) {
            const requiredFields = form.querySelectorAll("[required]");
            let valid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.style.borderColor = "#dc3545";
                    valid = false;
                } else {
                    field.style.borderColor = "#ddd";
                }
            });
            
            if (!valid) {
                e.preventDefault();
                alert("Please fill in all required fields");
            }
        });
    });
    
    // Auto-hide alerts after 5 seconds
    const alerts = document.querySelectorAll(".alert");
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.opacity = "0";
            setTimeout(() => alert.remove(), 300);
        }, 5000);
    });
});
';
}

function testSystem() {
    global $errors;
    
    try {
        // Test configuration loading
        if (!file_exists('config/app.php')) {
            $errors[] = 'Configuration file missing';
            return false;
        }
        
        // Test directory permissions
        $testFile = 'storage/data/test.txt';
        if (!file_put_contents($testFile, 'test')) {
            $errors[] = 'Cannot write to storage directory';
            return false;
        }
        unlink($testFile);
        
        // Test JSON operations
        $testData = ['test' => 'data'];
        $testJson = 'storage/data/test.json';
        if (!file_put_contents($testJson, json_encode($testData))) {
            $errors[] = 'JSON operations failed';
            return false;
        }
        
        $loadedData = json_decode(file_get_contents($testJson), true);
        if ($loadedData['test'] !== 'data') {
            $errors[] = 'JSON data integrity test failed';
            return false;
        }
        unlink($testJson);
        
        return true;
        
    } catch (Exception $e) {
        $errors[] = 'System test failed: ' . $e->getMessage();
        return false;
    }
}

// Check if already installed
$isInstalled = file_exists('INSTALLATION_COMPLETE');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Elevadex Installation</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            margin: 0;
            padding: 20px;
            min-height: 100vh;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }
        
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .header h1 {
            color: #333;
            margin: 0 0 10px 0;
        }
        
        .header p {
            color: #666;
            margin: 0;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            font-weight: 600;
            margin-bottom: 5px;
            color: #333;
        }
        
        .form-control {
            width: 100%;
            padding: 12px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s ease;
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border-left: 4px solid #28a745;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border-left: 4px solid #dc3545;
        }
        
        .alert-warning {
            background: #fff3cd;
            color: #856404;
            border-left: 4px solid #ffc107;
        }
        
        .step {
            display: flex;
            align-items: center;
            padding: 15px;
            margin-bottom: 10px;
            border-radius: 8px;
            background: #f8f9fa;
        }
        
        .step.completed {
            background: #d4edda;
        }
        
        .step.failed {
            background: #f8d7da;
        }
        
        .step.running {
            background: #fff3cd;
        }
        
        .step-icon {
            margin-right: 15px;
            font-size: 20px;
        }
        
        .already-installed {
            text-align: center;
            padding: 40px;
        }
        
        .already-installed h2 {
            color: #28a745;
            margin-bottom: 20px;
        }
        
        .grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        @media (max-width: 768px) {
            .grid {
                grid-template-columns: 1fr;
            }
            
            .container {
                padding: 20px;
                margin: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if ($isInstalled): ?>
            <div class="already-installed">
                <h2>✅ Installation Complete</h2>
                <p>Elevadex is already installed and ready to use.</p>
                <div style="margin-top: 30px;">
                    <a href="admin/" class="btn">Go to Admin Panel</a>
                </div>
                <p style="margin-top: 20px; font-size: 14px; color: #666;">
                    Installation completed on: <?php echo file_get_contents('INSTALLATION_COMPLETE'); ?>
                </p>
            </div>
        <?php else: ?>
            <div class="header">
                <h1>🚀 Elevadex Installation</h1>
                <p>Welcome! Let's get your Review Management System up and running.</p>
            </div>

            <?php if (!empty($errors)): ?>
                <?php foreach ($errors as $error): ?>
                    <div class="alert alert-error">❌ <?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <?php if (!empty($warnings)): ?>
                <?php foreach ($warnings as $warning): ?>
                    <div class="alert alert-warning">⚠️ <?php echo htmlspecialchars($warning); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <?php if (!empty($success)): ?>
                <?php foreach ($success as $successMsg): ?>
                    <div class="alert alert-success">✅ <?php echo htmlspecialchars($successMsg); ?></div>
                <?php endforeach; ?>
            <?php endif; ?>

            <?php if ($_SERVER['REQUEST_METHOD'] === 'POST'): ?>
                <h3>Installation Progress</h3>
                <?php foreach ($installationSteps as $step): ?>
                    <div class="step <?php echo $step['status']; ?>">
                        <span class="step-icon">
                            <?php
                            switch ($step['status']) {
                                case 'completed': echo '✅'; break;
                                case 'failed': echo '❌'; break;
                                case 'running': echo '⏳'; break;
                                default: echo '⭕'; break;
                            }
                            ?>
                        </span>
                        <span><?php echo htmlspecialchars($step['name']); ?></span>
                    </div>
                <?php endforeach; ?>

                <?php if (empty($errors)): ?>
                    <div style="text-align: center; margin-top: 30px;">
                        <h3 style="color: #28a745;">🎉 Installation Successful!</h3>
                        <p>Your Elevadex Review Management System is now ready to use.</p>
                        <a href="admin/" class="btn">Access Admin Panel</a>
                    </div>
                <?php endif; ?>

            <?php else: ?>
                <form method="POST">
                    <input type="hidden" name="action" value="install">
                    
                    <h3>Configuration Settings</h3>
                    
                    <div class="grid">
                        <div class="form-group">
                            <label for="site_name" class="form-label">Site Name</label>
                            <input type="text" id="site_name" name="site_name" class="form-control" 
                                   value="Elevadex Review Management System" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="site_url" class="form-label">Site URL</label>
                            <input type="url" id="site_url" name="site_url" class="form-control" 
                                   value="<?php echo 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['PHP_SELF']), '/'); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_email" class="form-label">Admin Email</label>
                            <input type="email" id="admin_email" name="admin_email" class="form-control" 
                                   value="admin@<?php echo $_SERVER['HTTP_HOST']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_password" class="form-label">Admin Password</label>
                            <input type="password" id="admin_password" name="admin_password" class="form-control" 
                                   placeholder="Enter a secure password (min 6 characters)" required minlength="6">
                        </div>
                    </div>
                    
                    <div style="text-align: center; margin-top: 30px;">
                        <button type="submit" class="btn">🚀 Install Elevadex</button>
                    </div>
                </form>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</body>
</html>